//
//  MPPushNotification.h
//  MPServices
//
//  Created by Oleksii Dykan on 4/15/14.
//  Copyright (c) 2014 MacPhun. All rights reserved.
//

#import <Foundation/Foundation.h>

/**
 MPPushNotification class is used to handle Macphun Push Notification
 
 The full required code to handle Macphun Push Notifications
 
    @code
    - (void)applicationDidFinishLaunching:(NSNotification *)notification {
        [MPPushNotification start:notification];
    }
 
    - (void)application:(NSApplication *)application didRegisterForRemoteNotificationsWithDeviceToken:(NSData *)deviceToken {
        [[NSUserNotificationCenter defaultUserNotificationCenter] setDelegate:self];
 
        [MPPushNotification sendStatsWithToken:deviceToken];
    }
 
    - (void)userNotificationCenter:(NSUserNotificationCenter *)center didActivateNotification:(NSUserNotification *)notification {
        [MPPushNotification handleNotification:notification];
    }
 
    - (void)userNotificationCenter:(NSUserNotificationCenter *)center didDeliverNotification:(NSUserNotification *)notification {
        [MPPushNotification deliveredNotification:notification];
    }
    @endcode
 
 @see '+start:'
 @see '+sendStatsWithToken:'
 @see '+deliveredNotification:'
 @see '+handleNotification:'
 
*/

typedef void(^MPPushNotificationHandler)(NSString *value);

@interface MPPushNotification : NSObject

/**
 Start handling push notifications
 
 Start handling Macphun Push Notifications. Should be implemented in app delegate applicationDidFinishLaunching: method
 
    @code
    - (void)applicationDidFinishLaunching:(NSNotification *)notification {
    [MPPushNotification start:notification];
    }
    @endcode
 
 @param notification Notification that comes from applicationDidFinishLaunching: method
 */
+ (void)start:(NSNotification *)notification;

/**
 This method is deprecated!
 
 Register a device token for a specified app name
 
    @code
    - (void)application:(NSApplication *)application didRegisterForRemoteNotificationsWithDeviceToken:(NSData *)deviceToken {
        [[NSUserNotificationCenter defaultUserNotificationCenter] setDelegate:self];
 
        [MPPushNotification registerDeviceToken:deviceToken appName:kPushNotificationAppName];
    }
    @endcode
 
 @param deviceToken Push notification device token
 @param appName Name of the current app
 @see '+sendStatsWithToken:'
*/
+ (void)registerDeviceToken:(NSData *)deviceToken appName:(NSString *)appName __attribute__ ((deprecated("use '+sendStatsWithToken:' instead.")));

/**
 Send statistics with specified token.
 
 This method is used to register a specified device token
 
    @code
    - (void)application:(NSApplication *)application didRegisterForRemoteNotificationsWithDeviceToken:(NSData *)deviceToken {
        [[NSUserNotificationCenter defaultUserNotificationCenter] setDelegate:self];
 
        [MPPushNotification sendStatsWithToken:deviceToken];
    }
    @endcode
 
 @param deviceToken Specified device token
 */
+ (void)sendStatsWithToken:(NSData *)deviceToken;

/**
 Handle delivery of notification
 @warning Should be implemented only in NSUserNotification Delegate method userNotificationCenter:didDeliverNotification:
 
    @code
    - (void)userNotificationCenter:(NSUserNotificationCenter *)center didDeliverNotification:(NSUserNotification *)notification {
        [MPPushNotification deliveredNotification:notification];
    }
    @endcode
 
 @param notification Delivered notification.
 */
+ (void)deliveredNotification:(NSUserNotification *)notification;

/**
 Handle activation of notification
 @warning Should be implemented only in NSUserNotification Delegate method userNotificationCenter:didActivateNotification:
 
    @code
    - (void)userNotificationCenter:(NSUserNotificationCenter *)center didActivateNotification:(NSUserNotification *)notification {
        [MPPushNotification handleNotification:notification];
    }
    @endcode
 
 @param notification Delivered notification.
 */
+ (void)handleNotification:(NSUserNotification *)notification;

/**
 Add custom push notification handler for specific type of notification
 
 @param handler The block for handling the reading of the notification. The block takes values string as the parameter
 @param type The type of push notification specified in push notification json
 */

+ (void)setNotificationHandler:(MPPushNotificationHandler)handler forType:(NSString *)type;

@end
